#include <fstream>
#include "GIFreader.h"
#pragma pack(1)

void GIFreader::AddFrame(GIFframe* const newframe)
{
	GIFframe ** pTempImg = new GIFframe* [last_frame+1];
	for (unsigned int n = 0; n < last_frame; n++)
		pTempImg[n] = img[n];
	delete [] img;
	img = pTempImg;
	img[last_frame] = newframe;
	last_frame++;
}

void GIFframe::Init(const unsigned int _xres, const unsigned int _yres, const unsigned int _BPP)
{
	Transparent = ~0u;
	xres = _xres;
	yres = _yres;
	BPP = _BPP;
	xPos = yPos = Delay = 0;

	if (pic)
		delete [] pic;
	if (pal)
		delete [] pal;
	pic = new unsigned char [xres*yres];
	pal = new COLORREF[1<<BPP];
}

bool GIFreader::Get(unsigned int * const pic, const unsigned int frame)
{	
	const GIFframe * const f = img[frame];

	gif_frame_counter = frame+1;

	for(unsigned int y = 0; y < f->yres; y++)
		for(unsigned int x = 0; x < f->xres; x++) if(!f->Transparency || (f->Transparent != f->pic[x+y*f->xres]))
			pic[x+f->xPos + (y+f->yPos)*xres] = (f->Transparent == f->pic[x+y*f->xres]) ? 0xFF000000 : f->pal[f->pic[x+y*f->xres]];

	return true;
}

bool GIFreader::Get(unsigned int * const pic)
{	
	return Get(pic, gif_frame_counter);
}

int LZWDecoder (const unsigned char * const bufIn, unsigned char * const bufOut,
				const short InitCodeSize, const unsigned int AlignedWidth,
				const unsigned int Width, const unsigned int Height, const int Interlace)
{
	unsigned int row=0,col=0;				// used to point output if Interlaced
	unsigned int nPixels, maxPixels;		// Output pixel counter

	unsigned int CodeSize;					// Current CodeSize (size in bits of codes)
	unsigned int ClearCode;					// Clear code : resets decompressor
	unsigned int EndCode;					// End code : marks end of information

	unsigned int whichBit;					// Index of next bit in bufIn
	unsigned int LongCode;					// Temp. var. from which Code is retrieved
	unsigned int Code;						// Code extracted
	unsigned int PrevCode;					// Previous Code
	unsigned int OutCode;					// Code to output

	// Translation Table:
	unsigned int Prefix[65536];				// Prefix: index of another Code
	unsigned int Suffix[65536];				// Suffix: terminating character
	unsigned int FirstEntry;				// Index of first free entry in table
	unsigned int NextEntry;					// Index of next free entry in table

	unsigned int OutStack[65537];			// Output buffer
	unsigned int OutIndex;					// Characters in OutStack

	unsigned int RowOffset;					// Offset in output buffer for current row

	// Set up values that depend on InitCodeSize Parameter.
	CodeSize = InitCodeSize+1;
	ClearCode = (1u << InitCodeSize);
	EndCode = ClearCode + 1;
	NextEntry = FirstEntry = ClearCode + 2;

	whichBit  = 0;
	nPixels   = 0;
	maxPixels = Width*Height;
	RowOffset = 0;

	while (nPixels<maxPixels) {
		OutIndex = 0;							// Reset Output Stack

		// GET NEXT CODE FROM bufIn:
		// LZW compression uses code items longer than a single byte.
		// For GIF Files, code sizes are variable between 9 and 12 bits 
		// That's why we must read data (Code) this way:
		LongCode=*((unsigned long*)(bufIn+whichBit/8));	// Get some bytes from bufIn
		LongCode>>=(whichBit&7);				// Discard too low bits
		Code =(LongCode & ((1u<<CodeSize)-1) );	// Discard too high bits
		whichBit += CodeSize;					// Increase Bit Offset

		// SWITCH, DIFFERENT POSIBILITIES FOR CODE:
		if (Code == EndCode)					// END CODE
			break;								// Exit LZW Decompression loop

		if (Code == ClearCode) {				// CLEAR CODE:
			CodeSize = InitCodeSize+1;			// Reset CodeSize
			NextEntry = FirstEntry;				// Reset Translation Table
			PrevCode=Code;				// Prevent next to be added to table.
			continue;							// restart, to get another code
		}
		if (Code < NextEntry)					// CODE IS IN TABLE
			OutCode = Code;						// Set code to output.

		else {									// CODE IS NOT IN TABLE:
			OutIndex++;			// Keep "first" character of previous output.
			OutCode = PrevCode;					// Set PrevCode to be output
		}

		// EXPAND OutCode IN OutStack
		// - Elements up to FirstEntry are Raw-Codes and are not expanded
		// - Table Prefices contain indexes to other codes
		// - Table Suffices contain the raw codes to be output
		while (OutCode >= FirstEntry) {
			if (OutIndex > 65536)
				return 0;
			OutStack[OutIndex++] = Suffix[OutCode];	// Add suffix to Output Stack
			OutCode = Prefix[OutCode];				// Loop with preffix
		}

		// NOW OutCode IS A RAW CODE, ADD IT TO OUTPUT STACK.
		if (OutIndex > 65536)
			return 0;
		OutStack[OutIndex++] = OutCode;

		// ADD NEW ENTRY TO TABLE (PrevCode + OutCode)
		// (EXCEPT IF PREVIOUS CODE WAS A CLEARCODE)
		if (PrevCode!=ClearCode) {
			Prefix[NextEntry] = PrevCode;
			Suffix[NextEntry] = OutCode;
			NextEntry++;

			// Prevent Translation table overflow:
			if (NextEntry >= 65536)
				return 0;
      
			// INCREASE CodeSize IF NextEntry IS INVALID WITH CURRENT CodeSize
			if (NextEntry >= (1u<<CodeSize)) {
				if (CodeSize < 12)
					CodeSize++;
				else {}				// Do nothing. Maybe next is Clear Code.
			}
		}

		PrevCode = Code;

		// Avoid the possibility of overflow on 'bufOut'.
		if (nPixels + OutIndex > maxPixels)
			OutIndex = maxPixels-nPixels;

		// OUTPUT OutStack (LAST-IN FIRST-OUT ORDER)
		for (int n=OutIndex-1; n>=0; n--) {
			if (col==Width)						// Check if new row.
			{
				if (Interlace) {				// If interlaced::
					     if ((row&7)==0) {row+=8; if (row>=Height) row=4;}
					else if ((row&3)==0) {row+=8; if (row>=Height) row=2;}
					else if ((row&1)==0) {row+=4; if (row>=Height) row=1;}
					else row+=2;
				}
				else							// If not interlaced:
					row++;

				RowOffset=row*AlignedWidth;		// Set new row offset
				col=0;
			}
			bufOut[RowOffset+col]=OutStack[n];	// Write output
			col++;	nPixels++;					// Increase counters.
		}
	}

	return whichBit;
}

int GIFreader::Open(const char * const szFileName)
{
	struct GIFGCEtag {				// GRAPHIC CONTROL EXTENSION
		unsigned char BlockSize;		// Block Size: 4 bytes
		unsigned char PackedFields;		// Packed Fields. Bits detail:
										//    0: Transparent Color Flag
										//    1: User Input Flag
										//  2-4: Disposal Method
		unsigned short Delay;			// Delay Time (1/100 seconds)
		unsigned char Transparent;		// Transparent Color Index
	} gifgce;
	int GraphicExtensionFound = 0;

	// OPEN FILE
	std::ifstream giffile (szFileName,std::ios::in | std::ios::binary);
	if (!giffile.is_open())
		return 0;

	// *1* READ HEADER (SIGNATURE + VERSION)
	char szSignature[6];				// First 6 bytes (GIF87a or GIF89a)
	giffile.read(szSignature,6);
	if ( memcmp(szSignature,"GIF",2) != 0)
		return 0;

	// *2* READ LOGICAL SCREEN DESCRIPTOR
	struct GIFLSDtag {
		unsigned short ScreenWidth;		// Logical Screen Width
		unsigned short ScreenHeight;	// Logical Screen Height
		unsigned char PackedFields;		// Packed Fields. Bits detail:
										//  0-2: Size of Global Color Table
										//    3: Sort Flag
										//  4-6: Color Resolution
										//    7: Global Color Table Flag
		unsigned char Background;		// Background Color Index
		unsigned char PixelAspectRatio;	// Pixel Aspect Ratio
	} giflsd;

	giffile.read((char*)&giflsd,sizeof(giflsd));

	const unsigned int GlobalBPP = (giflsd.PackedFields & 0x07) + 1;

	// fill some animation data:
	xres = giflsd.ScreenWidth;
	yres = giflsd.ScreenHeight;

	// *3* READ/GENERATE GLOBAL COLOR MAP
	COLORREF * const GlobalColorMap = new COLORREF[1<<GlobalBPP];
	if (giflsd.PackedFields & 0x80)	// file has global color map?
		for (unsigned int n = 0; n < 1u<<GlobalBPP; n++)
			GlobalColorMap[n] = (unsigned int)giffile.get() | (unsigned int)giffile.get()<<8 | (unsigned int)giffile.get()<<16;
	else	// default palette
		for (unsigned int n = 0; n < 256; n++)
			GlobalColorMap[n] = n | n<<8 | n<<16;

	// *4* NOW WE HAVE 3 POSSIBILITIES:
	//  4a) Get and Extension Block (Blocks with additional information)
	//  4b) Get an Image Separator (Introductor to an image)
	//  4c) Get the trailer Char (End of GIF File)
	do
	{
		int charGot = giffile.get();

		if (charGot == 0x21)		// *A* EXTENSION BLOCK 
		{
			switch (giffile.get())
			{

			case 0xF9:			// Graphic Control Extension

				giffile.read((char*)&gifgce,sizeof(gifgce));
				GraphicExtensionFound++;
				giffile.get(); // Block Terminator (always 0)
				break;

			case 0xFE:			// Comment Extension: Ignored
			case 0x01:			// PlainText Extension: Ignored
			case 0xFF:			// Application Extension: Ignored
			default:			// Unknown Extension: Ignored
				// read (and ignore) data sub-blocks
				while (unsigned int nBlockLength = giffile.get())
					for (unsigned int n=0; n < nBlockLength; n++) giffile.get();
				break;
			}
		}

		else if (charGot == 0x2c) {	// *B* IMAGE (0x2c Image Separator)

			// Create a new Image Object:
			GIFframe* NextImage = new GIFframe;

			// Read Image Descriptor
			struct GIFIDtag {	
				unsigned short xPos;			// Image Left Position
				unsigned short yPos;			// Image Top Position
				unsigned short Width;			// Image Width
				unsigned short Height;			// Image Height
				unsigned char PackedFields;		// Packed Fields. Bits detail:
											//  0-2: Size of Local Color Table
											//  3-4: (Reserved)
											//    5: Sort Flag
											//    6: Interlace Flag
											//    7: Local Color Table Flag
			} gifid;

			giffile.read((char*)&gifid, sizeof(gifid));

			int LocalColorMap = (gifid.PackedFields & 0x08)? 1 : 0;

			NextImage->Init (gifid.Width, gifid.Height,
				LocalColorMap ? (gifid.PackedFields&7)+1 : GlobalBPP);

			// Fill NextImage Data
			NextImage->xPos = gifid.xPos;
			NextImage->yPos = gifid.yPos;
			if (GraphicExtensionFound)
			{
				NextImage->Transparent=
					(gifgce.PackedFields&0x01) ? gifgce.Transparent : ~0u;
				NextImage->Transparency=
					(gifgce.PackedFields&0x1c)>1 ? 1 : 0;
				NextImage->Delay = gifgce.Delay*10;
			}
		
			if (LocalColorMap)		// Read Color Map (if descriptor says so)
				giffile.read((char*)NextImage->pal,
					sizeof(COLORREF)*(1<<NextImage->BPP));

			else					// Otherwise copy Global
				memcpy (NextImage->pal, GlobalColorMap,
					sizeof(COLORREF)*(1<<NextImage->BPP));

			short firstbyte=giffile.get();	// 1st byte of img block (CodeSize)

			// Calculate compressed image block size
				// to fix: this allocates an extra byte per block
			long ImgStart,ImgEnd;				
			ImgEnd = ImgStart = giffile.tellg();
			while (unsigned int n=giffile.get()) giffile.seekg (ImgEnd+=n+1);
			giffile.seekg (ImgStart);

			// Allocate Space for Compressed Image
			unsigned char * pCompressedImage = new unsigned char [ImgEnd-ImgStart+4];
  
			// Read and store Compressed Image
			unsigned char * pTemp = pCompressedImage;
			while (int nBlockLength = giffile.get())
			{
				giffile.read((char*)pTemp,nBlockLength);
				pTemp+=nBlockLength;
			}

			// Call LZW/GIF decompressor
			const unsigned int n=LZWDecoder(
				pCompressedImage,
				NextImage->pic,
				firstbyte, NextImage->xres,
				gifid.Width, gifid.Height,
				(gifid.PackedFields & 0x40) ? 1 : 0	//Interlaced?
				);

			if (n)
				AddFrame(NextImage);
			else
				delete NextImage;

			// Some cleanup
			delete [] pCompressedImage;
			GraphicExtensionFound = 0;
		}

		else if (charGot == 0x3b) {	// *C* TRAILER: End of GIF Info
			break; // Ok. Standard End.
		}

	} while (giffile.good());

	giffile.close();
	return last_frame;
}
