#ifdef WIN32
 #include <windows.h>
 #include <atlimage.h>
 #include "AVIreader.h"
 #include "GIFreader.h"
#else
 #include <stdlib.h>
 #include <memory.h>
#endif
#ifdef WIN32
 #include <GL/gl.h>
 #include <GL/glu.h>
 #include "glut.h"
#elif MACOSX
 #include <OpenGL/gl.h>
 #include <OpenGL/glu.h>
 #include <GLUT/glut.h>
#else
 #include <GL/gl.h>
 #include <GL/glu.h>
 #include <GL/glut.h>
#endif
#include <math.h>
#include <stdio.h>

// Milliseconds per frame to delay during animations
#define ANIMATION_DELAY 25

// Use high quality Stucki or otherwise Floyd Steinberg for dithering
#define STUCKI_DITHERING

// Use a jittered Bayer pattern for animation ditherung (slightly decreased compression rate but better quality for moving pictures (at 10+ fps))
#define JITTER_AVI_ORDERED_DITHERING
// Bayer 8x8 ordered dithering mask (normally used for b/w pictures, but tweaked implementation allows for more values)
const int bayer_dither_pattern[8][8] = {
    { 0, 32,  8, 40,  2, 34, 10, 42},   
    {48, 16, 56, 24, 50, 18, 58, 26},   
    {12, 44,  4, 36, 14, 46,  6, 38},   
    {60, 28, 52, 20, 62, 30, 54, 22},   
    { 3, 35, 11, 43,  1, 33,  9, 41},   
    {51, 19, 59, 27, 49, 17, 57, 25},
    {15, 47,  7, 39, 13, 45,  5, 37},
    {63, 31, 55, 23, 61, 29, 53, 21} };


// DMD resolution
int xres;
int yres;

// Window resolution
int winxres;
int winyres;

// Current DMD page (not zero only for animations)
int current_page = 0;
// For Playback
int first_page = 0;
int last_page = 0;
// Maximum numer of DMD pages (larger one only for animations)
int max_pages = 1;

// Raw DMD pixels
unsigned char **pic;
// Fancy DMD pixels (only used for blitting)
unsigned int *blitpic;

// Undo/Redo Buffering
#define MAX_UNDOREDO 32
unsigned char **undoredobuffer;
int *undo;
int *redo;

// Drawing Cursor
int cursorx;
int cursory;

// Color Mapping (32Bit Color)
unsigned int numcol;
unsigned int *invcol;
unsigned int **col;

// DMDs do not have real black but rather some grayish cells, this is the ratio to 100 
unsigned int DMDmincol;

// Currently active drawing color
unsigned int ccol;

// Lockmode to draw with cursor keys
bool lockdraw = false;

// Playback
bool replay = false;

// Stencil/Sprite
bool stencil = false;
unsigned int stencilcol;
const unsigned int STENCIL_OUT[2] = {0xFF00CC,0xCC00FF};

// DMD/pixel drawing mode
int DMDdisplay = 1;

// DMD color scheme
int DMDcolors = 0;

// Texture for blitting
#define GL_BGR_EXT  0x80E0
#define GL_BGRA_EXT 0x80E1
GLuint TexID;
int texxres;
int texyres;

// Blit Texture
void drawTex()
{	
	// Select Texture
	glBindTexture(GL_TEXTURE_2D, TexID);

	// Choose Bilerp Filter or No Filter (if there is no scaling of the picture)
	if((winxres-1 != xres*4) || (winyres-1 != yres*4))
	{
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	}
	else
	{
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	}

	// Upload Texture
	glTexSubImage2D(GL_TEXTURE_2D, 0, 0, 0, xres*4, yres*4,	GL_BGRA_EXT, GL_UNSIGNED_BYTE, blitpic);

	glClear(GL_COLOR_BUFFER_BIT);

	// Blit Fullscreen-Poly
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluOrtho2D(0.0,xres*4-1,0.0,yres*4-1);

	glColor4f(1.0f,1.0f,1.0f,1.0f);
	glBegin(GL_QUADS);
	glTexCoord2d(0.0,0.0);
	glVertex2d(1,yres*4-1);
	glTexCoord2d(xres/(double)texxres,0.0);
	glVertex2d(xres*4-1,yres*4-1);
	glTexCoord2d(xres/(double)texxres,yres/(double)texyres);
	glVertex2d(xres*4-1,1);
	glTexCoord2d(0.0,yres/(double)texyres);
	glVertex2d(1,1);
	glEnd(); 

	// Blit Cursor
	const float t = fabs(sin(3.14159265*2.0*((glutGet(GLUT_ELAPSED_TIME)>>3)&255)/255.0));
    glDisable(GL_TEXTURE_2D);
	glColor4f(t,t,t,1.0f);
	glBegin(GL_LINE_LOOP);
	glVertex2d(cursorx*4-1,(yres-1-cursory)*4-1);
	glVertex2d(cursorx*4-1,(yres-1-cursory)*4+4);
	glVertex2d(cursorx*4+4,(yres-1-cursory)*4+4);
	glVertex2d(cursorx*4+4,(yres-1-cursory)*4-1);
	glEnd();

	glEnable(GL_TEXTURE_2D);
}

unsigned int GS(const unsigned int RGB)
{
	const float r = (RGB>>16)&255;
	const float g = (RGB>>8)&255;
	const float b = RGB&255;

	return	(RGB>>24) ? invcol[stencilcol]*255/100 :
		    //(0.2126f*r + 0.7152f*g + 0.0724f*b); //Photoshop
			//(0.299f*r + 0.587f*g + 0.114f*b); //NTSC
			//(0.28f*r + 0.59f*g + 0.13f*b); //Flicker Test
			(0.27f*r + 0.67f*g + 0.06f*b); //Visually Based
}

unsigned int GS2(const unsigned int RGB)
{
	const float b = (RGB>>16)&255;
	const float g = (RGB>>8)&255;
	const float r = RGB&255;

	return	(RGB>>24) ? invcol[stencilcol]*255/100 :
		    //(0.2126f*r + 0.7152f*g + 0.0724f*b); //Photoshop
			//(0.299f*r + 0.587f*g + 0.114f*b); //NTSC
			//(0.28f*r + 0.59f*g + 0.13f*b); //Flicker Test
			(0.27f*r + 0.67f*g + 0.06f*b); //Visually Based
}

unsigned int MapColor(const int GS)
{
	for(unsigned int i = 0; i < numcol-1; i++)
		if(GS <= ((int)invcol[i]*255/100+(int)invcol[i+1]*255/100)/2) //!! TODO?
			return i;

	return numcol-1;
}

void TexInit()
{
	texxres = 1;
	while(texxres < xres)
		texxres *= 2;
	texyres = 1;
	while(texyres < yres)
		texyres *= 2;

	// Generate a Texture
	glEnable(GL_TEXTURE_2D);

	glGenTextures(1, &TexID);             
	glBindTexture(GL_TEXTURE_2D, TexID);

	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP); 
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);

	char * const temppic = (char*)malloc(texxres*4*texyres*4*4);
	// Upload Texture
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB8, texxres*4, texyres*4, 0,
		GL_BGRA_EXT, GL_UNSIGNED_BYTE, temppic);
	free(temppic);
}

void saveundostep()
{
	// If Buffer full then delete first entry (slow, but shouldn't matter on anything newer than a 486 ;)
	if(undo[current_page] == MAX_UNDOREDO-1)
	{
		memmove(undoredobuffer[current_page],undoredobuffer[current_page]+xres*yres,xres*yres*(MAX_UNDOREDO-1));
		undo[current_page]--;
		redo[current_page]--;
	}
	undo[current_page]++;
	redo[current_page] = undo[current_page];
	memcpy(undoredobuffer[current_page]+undo[current_page]*xres*yres,pic[current_page],xres*yres);
}

void undostep()
{
	if(undo[current_page] < 0)
		return;
	
	// If Buffer full then delete first entry (slow, but shouldn't matter on anything newer than a 486 ;)
	if(undo[current_page] == MAX_UNDOREDO-1)
	{
		memmove(undoredobuffer[current_page],undoredobuffer[current_page]+xres*yres,xres*yres*(MAX_UNDOREDO-1));
		undo[current_page]--;
		redo[current_page]--;
	}
	// Save last Editing Step
	if(undo[current_page]+1 > redo[current_page])
		redo[current_page] = undo[current_page]+1;
	memcpy(undoredobuffer[current_page]+(undo[current_page]+1)*xres*yres,pic[current_page],xres*yres);
	
	memcpy(pic[current_page],undoredobuffer[current_page]+undo[current_page]*xres*yres,xres*yres);
	undo[current_page]--;
}

void redostep()
{
	if((redo[current_page] < 0) || (undo[current_page] == redo[current_page]))
		return;
	undo[current_page]++;
	memcpy(pic[current_page],undoredobuffer[current_page]+undo[current_page]*xres*yres,xres*yres);
}

void display()
{	
	// Playback?
	if(replay)
	{
#ifdef WIN32
		Sleep(ANIMATION_DELAY); //!! simple Linux hack?!
#endif
		current_page++;
		if(current_page > last_page)
			current_page = first_page;
	}

	// Make a nice and fancy "DMD" out of the pixels
	
	if(DMDdisplay == 0) //light version
	for(int j = 0; j < yres; j++)
		for(int i = 0; i < xres; i++)
			if(stencil && (pic[current_page][i+j*xres] == stencilcol))
				for(int jj = 0; jj < 4; jj++)
					for(int ii = 0; ii < 4; ii++)
						blitpic[i*4+ii+(j*4+jj)*xres*4] = STENCIL_OUT[(j^i)&1];
			else
			{
				blitpic[i*4+1+(j*4+1)*xres*4] = col[DMDcolors][pic[current_page][i+j*xres]];
				blitpic[i*4+(j*4)*xres*4] = blitpic[i*4+2+(j*4+2)*xres*4] = blitpic[i*4+2+(j*4)*xres*4] = blitpic[i*4+(j*4+2)*xres*4] = (col[DMDcolors][pic[current_page][i+j*xres]]&0xFCFCFCFC)>>2;
				blitpic[i*4+2+(j*4+1)*xres*4] = blitpic[i*4+1+(j*4+2)*xres*4] = blitpic[i*4+1+(j*4)*xres*4] = blitpic[i*4+(j*4+1)*xres*4] = (col[DMDcolors][pic[current_page][i+j*xres]]&0xFEFEFEFE)>>1;

				for(int jj = 0; jj < 4; jj++)
					blitpic[i*4+3+(j*4+jj)*xres*4] = 0;
				for(int ii = 0; ii < 4; ii++)
					blitpic[i*4+ii+(j*4+3)*xres*4] = 0;
			}
	else if(DMDdisplay == 1) //standard version
	for(int j = 0; j < yres; j++)
		for(int i = 0; i < xres; i++)
			if(stencil && (pic[current_page][i+j*xres] == stencilcol))
				for(int jj = 0; jj < 4; jj++)
					for(int ii = 0; ii < 4; ii++)
						blitpic[i*4+ii+(j*4+jj)*xres*4] = STENCIL_OUT[(j^i)&1];
			else
			{
				blitpic[i*4+1+(j*4+1)*xres*4] = col[DMDcolors][pic[current_page][i+j*xres]];
				blitpic[i*4+(j*4+1)*xres*4] = blitpic[i*4+2+(j*4+1)*xres*4] = blitpic[i*4+1+(j*4)*xres*4] = blitpic[i*4+1+(j*4+2)*xres*4] = ((col[DMDcolors][pic[current_page][i+j*xres]]&0xFCFCFCFC)*3)>>2;
				blitpic[i*4+(j*4)*xres*4] = blitpic[i*4+2+(j*4)*xres*4] = blitpic[i*4+(j*4+2)*xres*4] = blitpic[i*4+2+(j*4+2)*xres*4] = ((col[DMDcolors][pic[current_page][i+j*xres]]&0xF8F8F8F8)*3)>>3;

				for(int jj = 0; jj < 4; jj++)
					blitpic[i*4+3+(j*4+jj)*xres*4] = 0;
				for(int ii = 0; ii < 4; ii++)
					blitpic[i*4+ii+(j*4+3)*xres*4] = 0;
			}
	else if(DMDdisplay == 2) //blocky version
	for(int j = 0; j < yres; j++)
		for(int i = 0; i < xres; i++)
			if(stencil && (pic[current_page][i+j*xres] == stencilcol))
				for(int jj = 0; jj < 4; jj++)
					for(int ii = 0; ii < 4; ii++)
						blitpic[i*4+ii+(j*4+jj)*xres*4] = STENCIL_OUT[(j^i)&1];
			else
			{
				blitpic[i*4+1+(j*4+1)*xres*4] = blitpic[i*4+2+(j*4+1)*xres*4] = blitpic[i*4+1+(j*4+2)*xres*4] = blitpic[i*4+2+(j*4+2)*xres*4] = col[DMDcolors][pic[current_page][i+j*xres]];

				blitpic[i*4+(j*4+1)*xres*4] = blitpic[i*4+(j*4+2)*xres*4] = blitpic[i*4+3+(j*4+1)*xres*4] = blitpic[i*4+3+(j*4+2)*xres*4] = blitpic[i*4+1+(j*4)*xres*4] = blitpic[i*4+2+(j*4)*xres*4] = blitpic[i*4+1+(j*4+3)*xres*4] = blitpic[i*4+2+(j*4+3)*xres*4] = (col[DMDcolors][pic[current_page][i+j*xres]]&0xFEFEFEFE)>>1;
				
				blitpic[i*4+(j*4)*xres*4] = blitpic[i*4+3+(j*4+3)*xres*4] = blitpic[i*4+3+(j*4)*xres*4] = blitpic[i*4+(j*4+3)*xres*4] = (col[DMDcolors][pic[current_page][i+j*xres]]&0xFCFCFCFC)>>2;
			}
	else //pixels
	for(int j = 0; j < yres; j++)
		for(int i = 0; i < xres; i++)
			if(stencil && (pic[current_page][i+j*xres] == stencilcol))
				for(int jj = 0; jj < 4; jj++)
					for(int ii = 0; ii < 4; ii++)
						blitpic[i*4+ii+(j*4+jj)*xres*4] = STENCIL_OUT[(j^i)&1];
			else
				for(int jj = 0; jj < 4; jj++)
					for(int ii = 0; ii < 4; ii++)
						blitpic[i*4+ii+(j*4+jj)*xres*4] = col[DMDcolors][pic[current_page][i+j*xres]];
	
	// Draw it
	drawTex();
	char bla[256];
	if(((glutGet(GLUT_ELAPSED_TIME)>>5)&255) > 23)
		sprintf(bla,"=DMDpaint=   (page:%d/%d (active:%d/%d)   x:%d y:%d c:%d   drawcolor:%d)",current_page+1,max_pages,first_page+1,last_page+1,cursorx+1,cursory+1,pic[current_page][cursorx+cursory*xres]+1,ccol+1);
	else
		sprintf(bla,"=DMDpaint=   (right mouse button for help/menu)");
	glutSetWindowTitle(bla);
	glutSwapBuffers();
}

// Window size was updated
void reshape(int w, int h)
{
	if(w < 2)
		w = 2;
	if(h < 2)
		h = 2;
	glViewport(0, 0, w, h);  

	winxres = w;
	winyres = h;
}

// Flood fill
void recfill(const int x, const int y)
{
	if(pic[current_page][x+y*xres] == ccol)
		return;

	unsigned char oldcol = pic[current_page][x+y*xres];
	pic[current_page][x+y*xres] = ccol;
    if((x+1 < xres) && (oldcol == pic[current_page][x+1+y*xres]))
		recfill(x+1,y);
	if((y+1 < yres) && (oldcol == pic[current_page][x+(y+1)*xres]))
		recfill(x,y+1);
	if((x > 0) && (oldcol == pic[current_page][x-1+y*xres]))
		recfill(x-1,y);
	if((y > 0) && (oldcol == pic[current_page][x+(y-1)*xres]))
		recfill(x,y-1);
}

void keyboardSpecial(int key, int x, int y)
{
	// 'F1'-'F6' = select a color
	if(key == GLUT_KEY_F1)
		ccol = 10;
	if(key == GLUT_KEY_F2)
		ccol = 11;
	if(key == GLUT_KEY_F3)
		ccol = 12;
	if(key == GLUT_KEY_F4)
		ccol = 13;
	if(key == GLUT_KEY_F5)
		ccol = 14;
	if(key == GLUT_KEY_F6)
		ccol = 15;

	// 'LEFT',etc. = move cursor (and draw if lockdraw is enabled)
	if(key == GLUT_KEY_LEFT)
	{
		cursorx = (cursorx > 0) ? cursorx-1 : 0;
		if((lockdraw) && (pic[current_page][cursorx+cursory*xres] != ccol))
		{
			saveundostep();
			pic[current_page][cursorx+cursory*xres] = ccol;
		}
	}
	if(key == GLUT_KEY_RIGHT)
	{
		cursorx = (cursorx < xres-1) ? cursorx+1 : xres-1;
		if((lockdraw) && (pic[current_page][cursorx+cursory*xres] != ccol))
		{
			saveundostep();
			pic[current_page][cursorx+cursory*xres] = ccol;
		}
	}
	if(key == GLUT_KEY_UP)
	{
		cursory = (cursory > 0) ? cursory-1 : 0;
		if((lockdraw) && (pic[current_page][cursorx+cursory*xres] != ccol))
		{
			saveundostep();
			pic[current_page][cursorx+cursory*xres] = ccol;
		}
	}
	if(key == GLUT_KEY_DOWN)
	{
		cursory = (cursory < yres-1) ? cursory+1 : yres-1;
		if((lockdraw) && (pic[current_page][cursorx+cursory*xres] != ccol))
		{
			saveundostep();
			pic[current_page][cursorx+cursory*xres] = ccol;
		}
	}

	// PGUP = change display to show next page
	if(key == GLUT_KEY_PAGE_UP)
	{
		if(current_page < max_pages-1)
			current_page++;
		else
		{
			const int last_page_bak = last_page;

			if(last_page == max_pages-1)
				last_page++;
			max_pages++;

			pic = (unsigned char**)realloc(pic,sizeof(size_t)*max_pages);
			pic[max_pages-1] = (unsigned char*)malloc(xres*yres);
			if(pic[max_pages-1] == NULL) {
				last_page = last_page_bak;
				max_pages--;
				return;
			}
			memset(pic[max_pages-1],0,xres*yres);
			
			undoredobuffer = (unsigned char**)realloc(undoredobuffer,sizeof(size_t)*max_pages);
			undoredobuffer[max_pages-1] = (unsigned char*)malloc(xres*yres*MAX_UNDOREDO);
			if(undoredobuffer[max_pages-1] == NULL) {
				free(pic[max_pages-1]);
				last_page = last_page_bak;
				max_pages--;
				return;
			}

			undo = (int*)realloc(undo,sizeof(int)*max_pages);
			undo[max_pages-1] = 0;
			redo = (int*)realloc(redo,sizeof(int)*max_pages);
			redo[max_pages-1] = 0;

			current_page++;
		}
	}

	// PGDN = change display to show previous page
	if(key == GLUT_KEY_PAGE_DOWN)
	{
		if(current_page > 0)
			current_page--;
	}

	// HOME = set first DMD frame
	if(key == GLUT_KEY_HOME)
		first_page = current_page;

	// END = set last DMD frame
	if(key == GLUT_KEY_END)
		last_page = current_page;
}

void keyboard(unsigned char key, int x, int y)
{
	// ESC = exit
	if(key == 27)
		exit(0); //!! TODO

	// '+' = zoom window
	if(key == '+')
		glutReshapeWindow(glutGet(GLUT_WINDOW_WIDTH)*2,glutGet(GLUT_WINDOW_HEIGHT)*2);

	// '-' = zoom window
	if(key == '-')
		glutReshapeWindow(glutGet(GLUT_WINDOW_WIDTH)/2,glutGet(GLUT_WINDOW_HEIGHT)/2);

	// 'x' = stencil/sprite mode
	if(key == 'x')
		stencil = !stencil;

	// 'v' = DMD drawing scheme
	if(key == 'v')
		DMDdisplay = (DMDdisplay+1)%4;

	// 'n' = DMD color scheme
	if(key == 'n')
		DMDcolors = (DMDcolors+1)%4;

	// 'c' = clear screen with current color
	if(key == 'c')
	{
		saveundostep();
		for(int i = 0; i < xres*yres; i++)
			pic[current_page][i] = ccol;
	}

	// 's'/'d'/'a'/'b'/',' = save file
	if((key == 's') || (key == 'd') || (key == 'a') || (key == 'b') || (key == ','))
	{
		char szFile[1024];      // buffer for file name
		FILE *f;
#ifdef WIN32
		OPENFILENAME ofn;       // common dialog box structure

		// Initialize OPENFILENAME
		ZeroMemory(&ofn, sizeof(ofn));
		ofn.lStructSize = sizeof(ofn);
		ofn.lpstrFile = szFile;
		//
		// Set lpstrFile[0] to '\0' so that GetOpenFileName does not 
		// use the contents of szFile to initialize itself.
		//
		ofn.lpstrFile[0] = '\0';
		ofn.nMaxFile = sizeof(szFile);
		if(key == 'a')
		{
			ofn.lpstrFilter = "TXT\0*.txt\0";
			ofn.lpstrDefExt = "txt";
		} else if((key == 'b') || (key == ','))
		{
			ofn.lpstrFilter = "DMDF\0*.dmdf\0";
			ofn.lpstrDefExt = "dmdf";
		}
		else
		{
			ofn.lpstrFilter = "PGM\0*.pgm\0";
			ofn.lpstrDefExt = "pgm";
		}
		ofn.nFilterIndex = 1;
		ofn.lpstrFileTitle = NULL;
		ofn.nMaxFileTitle = 0;
		ofn.lpstrInitialDir = NULL;
		ofn.Flags = OFN_OVERWRITEPROMPT;

		if(!GetSaveFileName(&ofn))
			return;
#else
		if(key == 's')
			strcpy(szFile,"save.pgm"); //!! TODO
		if(key == 'd')
			strcpy(szFile,"dmdfreewpc.pgm"); //!! TODO
		if(key == 'a')
			strcpy(szFile,"dmd0123.txt"); //!! TODO
		if((key == 'b') || (key == ','))
			strcpy(szFile,"dmdf.dmdf"); //!! TODO
#endif
		unsigned int dmdfcounter = 0; // hacky way to store the 95 frames per file

		// loop over all (active) frames
		for(int p = first_page; p <= last_page; p++)
		{
			char szFile2[1024];
			strcpy(szFile2,szFile);

			// create six digits for filename (if more than one frame)
			if((key != 'a') && (last_page > first_page)) for(int d = 0; d < 6; d++)
			{
                const int sl = (int)strlen(szFile2);
				szFile2[sl+1] = szFile2[sl];
				szFile2[sl]   = szFile2[sl-1];
				szFile2[sl-1] = szFile2[sl-2];
				szFile2[sl-2] = szFile2[sl-3];
				szFile2[sl-3] = szFile2[sl-4];
				if((key == 'b') || (key == ',')) {
					szFile2[sl-4] = szFile2[sl-5];
					for(int d2 = 0; d2 < d; d2++)
						szFile2[sl-5-d2] = szFile2[sl-6-d2];
					szFile2[sl-5-d] = ((p/(int)pow(10.0,d))%10)+48; //!! TODO
				} else {
					for(int d2 = 0; d2 < d; d2++)
						szFile2[sl-4-d2] = szFile2[sl-5-d2];
					szFile2[sl-4-d] = ((p/(int)pow(10.0,d))%10)+48; //!! TODO
				}
			}

			f = fopen(szFile2,"w");
			if(f == NULL)
				return;

			if(key == 's')
			{
				fprintf(f,"P2\n# CREATOR: DMDpaint\n%d %d\n255\n",xres,yres);
				for(int i = 0; i < xres*yres; i++)
					fprintf(f,"%d\n",invcol[pic[p][i]]*255/100); //!! TODO?
			}
			if(key == 'd')
			{
				fprintf(f,"P2\n# CREATOR: DMDpaint\n%d %d\n255\n",xres,yres);
				for(int i = 0; i < xres*yres; i++)
					fprintf(f,"%d\n",((unsigned int)pic[p][i])*255/(numcol-1)); //!! TODO?
			}
			if((key == 'b') || (key == ','))
			{
				fclose(f);
				f = fopen(szFile2,"wb");
			
				struct {
					unsigned char name[8];
					unsigned char font_width;
					unsigned char font_width2;
					unsigned char font_height;
					unsigned char font_height2;
					unsigned char end[2];
				} dmdfheader;
				dmdfheader.name[0] = 'F';
				dmdfheader.name[1] = 'P';
				dmdfheader.name[2] = 'D';
				dmdfheader.name[3] = 'M';
				dmdfheader.name[4] = 'D';
				dmdfheader.name[5] = 'F';
				dmdfheader.name[6] = 'N';
				dmdfheader.name[7] = 'T';
				dmdfheader.font_width = (key == 'b') ? xres/19 : xres; //!! hardwired
				dmdfheader.font_width2 = 0;
				dmdfheader.font_height = (key == 'b') ? yres/5 : yres; //!! hardwired
				dmdfheader.font_height2 = 0;
				dmdfheader.end[0] = 0x5E;
				dmdfheader.end[1] = 0;
				fwrite(&dmdfheader,sizeof(dmdfheader),1,f);

				for(unsigned int c = 0; c < 95; ++c) {
				    struct {
					    unsigned char font_width;
					    unsigned char font_width2;
				    } dmdfheader2;
					dmdfheader2.font_width = dmdfheader.font_width;
					dmdfheader2.font_width2 = dmdfheader.font_width2;
				    fwrite(&dmdfheader2,sizeof(dmdfheader2),1,f);

					if(dmdfheader2.font_width > 0) {
 				        unsigned char * const tmp = (unsigned char*)malloc(dmdfheader2.font_width*dmdfheader.font_height);

						if(key == 'b') {
							for(unsigned int y = 0; y < dmdfheader.font_height; ++y) {
								unsigned int o = y*xres + (c*dmdfheader.font_width)%xres + ((c*dmdfheader.font_width)/xres * dmdfheader.font_height)*xres;
								for(unsigned int x = 0; x < dmdfheader2.font_width; ++x,++o) if(o < (unsigned int)(xres*yres)) 
									tmp[x+y*dmdfheader2.font_width] = (stencil && (pic[p][o] == stencilcol)) ? 7 : invcol[pic[p][o]]*4/100; // 7=transparent
							}
						}
						else if((int)(c+dmdfcounter)+first_page <= last_page) {
							unsigned int o = 0;
							for(unsigned int y = 0; y < dmdfheader.font_height; ++y)
								for(unsigned int x = 0; x < dmdfheader2.font_width; ++x,++o) 
									tmp[o] = (stencil && (pic[c+dmdfcounter+first_page][o] == stencilcol)) ? 7 : invcol[pic[c+dmdfcounter+first_page][o]]*4/100; // 7=transparent
					    } else
							memset(tmp,7,dmdfheader2.font_width*dmdfheader.font_height); // 7=transparent

						fwrite(tmp,1,dmdfheader2.font_width*dmdfheader.font_height,f);

				        free(tmp);
					}
				}

				dmdfcounter += 95;

				if((key != 'b') && (dmdfcounter >= last_page-first_page+1)) {
					fclose(f);
					return; // as all is stored in only one file
				}
			}
			if(key == 'a')
			{
				for(int p2 = first_page; p2 <= last_page; p2++)
				{
					for(int i = 0; i < xres*yres; i++)
					{			
						fprintf(f,"%d",pic[p2][i]);
						if((i%xres) == xres-1)
							fprintf(f,"\n");
					}
					if(p2 <= last_page-1)
						fprintf(f,"\n");
				}

				fclose(f);
				return; // as all is stored in only one file
			}
			
			fclose(f);
		}
	}

	// 'h' = load txt
	if(key == 'h')
	{
#ifdef WIN32
		OPENFILENAME ofn;       // common dialog box structure
		char szFile[1024];      // buffer for file name
		
		// Initialize OPENFILENAME
		ZeroMemory(&ofn, sizeof(ofn));
		ofn.lStructSize = sizeof(ofn);
		ofn.lpstrFile = szFile;
		//
		// Set lpstrFile[0] to '\0' so that GetOpenFileName does not 
		// use the contents of szFile to initialize itself.
		//
		ofn.lpstrFile[0] = '\0';
		ofn.nMaxFile = sizeof(szFile);
		ofn.lpstrFilter = "TXT\0*.txt\0";
		ofn.nFilterIndex = 1;
		ofn.lpstrFileTitle = NULL;
		ofn.nMaxFileTitle = 0;
		ofn.lpstrInitialDir = NULL;
		ofn.Flags = OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST;

		if (!GetOpenFileName(&ofn))
			return;
		
		FILE *f = fopen(szFile,"r");
#else
		FILE *f = fopen("test.txt","r"); //!! TODO
#endif
		if(f == NULL)
			return;

		const int old_page = current_page;
		char bla[64];

		while(true)
		{
			saveundostep();

			for(int j = 0; j < yres; j++)
			{
				for(int i = 0; i < xres; i++)
					pic[current_page][i+j*xres] = fgetc(f)-48; //!! TODO
				fgets(bla,64,f);
			}

			fgets(bla,64,f);

			if(feof(f))
				break;

			keyboardSpecial(GLUT_KEY_PAGE_UP,0,0); //!! TODO
		}

		current_page = old_page;

		fclose(f);
	}

	// 'l'/'k'/'j' = load picture
	if((key == 'l') || (key == 'k') || (key == 'j'))
	{
		unsigned int *pict;
#ifdef WIN32
		OPENFILENAME ofn;       // common dialog box structure
		char szFile[1024];      // buffer for file name
		
		// Initialize OPENFILENAME
		ZeroMemory(&ofn, sizeof(ofn));
		ofn.lStructSize = sizeof(ofn);
		ofn.lpstrFile = szFile;
		//
		// Set lpstrFile[0] to '\0' so that GetOpenFileName does not 
		// use the contents of szFile to initialize itself.
		//
		ofn.lpstrFile[0] = '\0';
		ofn.nMaxFile = sizeof(szFile);
		ofn.lpstrFilter = "All\0*.*\0PGM\0*.pgm\0DMDF\0*.dmdf\0";
		ofn.nFilterIndex = 1;
		ofn.lpstrFileTitle = NULL;
		ofn.nMaxFileTitle = 0;
		ofn.lpstrInitialDir = NULL;
		ofn.Flags = OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST;

		if (GetOpenFileName(&ofn))
		{
			int slen = (int)strlen(szFile);
			if((szFile[slen-1] == 'f') && (szFile[slen-2] == 'd') && (szFile[slen-3] == 'm') && (szFile[slen-4] == 'd'))
			{
				FILE *f = fopen(szFile,"rb");
				if(f == NULL)
					return;

				saveundostep();

				pict = (unsigned int*)malloc(xres*yres*4);
				memset(pict,0,xres*yres*4);

				struct {
					unsigned char name[8];
					unsigned char font_width;
					unsigned char font_width2;
					unsigned char font_height;
					unsigned char font_height2;
					unsigned char end[2];
				} dmdfheader;
				fread(&dmdfheader,sizeof(dmdfheader),1,f);
				if(dmdfheader.name[0] != 'F' || dmdfheader.name[1] != 'P' || dmdfheader.name[2] != 'D' || dmdfheader.name[3] != 'M' || dmdfheader.name[4] != 'D'
					|| dmdfheader.font_width == 0 || dmdfheader.font_height == 0) {
					fclose(f);
					return;
				}

				for(unsigned int c = 0; c < 95; ++c) {
				    struct {
					    unsigned char font_width;
					    unsigned char font_width2;
				    } dmdfheader2;
				    fread(&dmdfheader2,sizeof(dmdfheader2),1,f);

					if(dmdfheader2.font_width > 0) {
 				        unsigned char *tmp = (unsigned char*)malloc(dmdfheader2.font_width*dmdfheader.font_height);
				        fread(tmp,1,dmdfheader2.font_width*dmdfheader.font_height,f);

					    for(unsigned int y = 0; y < dmdfheader.font_height; ++y) {
						    unsigned int o = y*xres + (c*dmdfheader.font_width)%xres + ((c*dmdfheader.font_width)/xres * dmdfheader.font_height)*xres;
					        for(unsigned int x = 0; x < dmdfheader2.font_width; ++x,++o) if(o < (unsigned int)(xres*yres)) 
								pict[o] = (tmp[x+y*dmdfheader2.font_width] != 7) ? (unsigned int)tmp[x+y*dmdfheader2.font_width]*255/4 : invcol[stencilcol]*255/100; // 7=transparent
					    }

				        free(tmp);
					}
				}

				fclose(f);
			}
			else if((szFile[slen-1] == 'm') && (szFile[slen-2] == 'g') && (szFile[slen-3] == 'p'))
			{
				FILE *f = fopen(szFile,"r");
				if(f == NULL)
					return;

				saveundostep();

				char s[256];
				do
				{
				    fscanf(f,"%s",s); //!! TODO
				}while(s[0] != '3');
				fscanf(f,"%d",pic[current_page]);

				pict = (unsigned int*)malloc(xres*yres*4); 
				
				for(int i = 0; i < xres*yres; i++)
					fscanf(f,"%d",&(pict[i]));

				fclose(f);
			}
			else
			{
				CImage myimage;

				if(myimage.Load(szFile) < 0)
					return;

				if((myimage.GetWidth() != xres) || (myimage.GetHeight() != yres))
				{
					myimage.Destroy();
#ifdef WIN32
					MessageBox(NULL,"Resolution does not match the DMD, please change resolution via command line options","LoadPic",MB_OK|MB_ICONSTOP);
#endif
					return;
				}

				saveundostep();

				pict = (unsigned int*)malloc(xres*yres*4); 
			
				for(int j = 0; j < yres; j++)
					for(int i = 0; i < xres; i++)
						pict[i+j*xres] = GS2(myimage.GetPixel(i,j));
			
				myimage.Destroy();
			}
		}
		else
			return;
#else
		FILE *f = fopen("test.pgm","r"); //!! TODO
		if(f == NULL)
			return;

		saveundostep();

		char s[256];
		do
		{
		    fscanf(f,"%s",s); //!! TODO
		}while(s[0] != '3');
		fscanf(f,"%d",pic[current_page]);

		pict = (unsigned int*)malloc(xres*yres*4); 
		
		for(int i = 0; i < xres*yres; i++)
			fscanf(f,"%d",&(pict[i]));

		fclose(f);
#endif
		const int divcol = (numcol > 4) ? (int)(numcol/4) : 1;
		for(int y = 0; y < yres; y++)
			for(int x = 0; x < xres; x++)
			{
				const unsigned int i = x+y*xres;
				const unsigned int i2 = x+y*xres;

					if(key == 'k')
					{
						const int Diff = (bayer_dither_pattern[y&7][x&7]-31)/divcol;

						if((int)pict[i2]+Diff <= 0)
							pic[current_page][i] = MapColor(0);
						else
							if((int)pict[i2]+Diff >= 255)
								pic[current_page][i] = MapColor(255);
							else
								pic[current_page][i] = MapColor(pict[i2]+Diff);
					}
					else
					{
						pic[current_page][i] = MapColor(pict[i2]);

						if(key == 'l')
						{
#ifdef STUCKI_DITHERING
							const int Err = pict[i2]-invcol[pic[current_page][i]]*255/100;
							if(x < xres-1) pict[i2+1] += 8*Err/42;
							if(x < xres-2) pict[i2+2] += 4*Err/42;
							if(y < yres-1)
							{
								if(x > 1) pict[i2-2+xres] += 2*Err/42;
								if(x > 0) pict[i2-1+xres] += 4*Err/42;
								pict[i2+xres] += 8*Err/42;
								if(x < xres-1) pict[i2+1+xres] += 4*Err/42;
								if(x < xres-2) pict[i2+2+xres] += 2*Err/42;
								if(y < yres-2)
								{
									if(x > 1) pict[i2-2+xres*2] += Err/42;
									if(x > 0) pict[i2-1+xres*2] += 2*Err/42;
									pict[i2+xres*2] += 4*Err/42;
									if(x < xres-1) pict[i2+1+xres*2] += 2*Err/42;
									if(x < xres-2) pict[i2+2+xres*2] += Err/42;
								}
							}
#else					
							const int Err = pict[i2]-invcol[pic[current_page][i]]*255/100;
							if(x < xres-1) pict[i2+1] += 7*Err/16;
							if(y < yres-1)
							{	
								if(x > 0) pict[i2-1+xres] += 3*Err/16;
								pict[i2+xres] += 5*Err/16;
								if(x < xres-1) pict[i2+1+xres] += Err/16;
							}
#endif
						}
					}

			}

		free(pict);
	}

	// 'm' = Playback of all frames
	if(key == 'm')
		replay = !replay;

#ifdef WIN32
	// 'o'/'i'/'u' = load avi/animatedgif/dmdf file
	if((key == 'o') || (key == 'i') || (key == 'u'))
	{
		OPENFILENAME ofn;       // common dialog box structure
		char szFile[1024];      // buffer for file name
		
		// Initialize OPENFILENAME
		ZeroMemory(&ofn, sizeof(ofn));
		ofn.lStructSize = sizeof(ofn);
		ofn.lpstrFile = szFile;
		//
		// Set lpstrFile[0] to '\0' so that GetOpenFileName does not 
		// use the contents of szFile to initialize itself.
		//
		ofn.lpstrFile[0] = '\0';
		ofn.nMaxFile = sizeof(szFile);
		ofn.lpstrFilter = "All\0*.*\0AVI\0*.avi\0DIVX\0*.divx\0GIF\0*.gif\0DMDF\0*.dmdf\0";
		ofn.nFilterIndex = 1;
		ofn.lpstrFileTitle = NULL;
		ofn.nMaxFileTitle = 0;
		ofn.lpstrInitialDir = NULL;
		ofn.Flags = OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST;

		if (!GetOpenFileName(&ofn))
			return;

		int slen = (int)strlen(szFile);
		if((szFile[slen-1] == 'f') && (szFile[slen-2] == 'i') && (szFile[slen-3] == 'g'))
		{
		GIFreader f;
		f.Open(szFile);

		// code is just copy-pasted again for AVI below!

		if((f.xres != xres) || (f.yres != yres)) {
#ifdef WIN32
			MessageBox(NULL,"Resolution does not match the DMD, please change resolution via command line options","LoadPic",MB_OK|MB_ICONSTOP);
#endif
			return;
		}

		unsigned int * const pictorg = (unsigned int*)malloc(xres*yres*4); 
		unsigned int * const pict = (unsigned int*)malloc(xres*yres*4); 
		
		const unsigned int old_page = current_page;

		for(int n = 0; n < (int)f.last_frame/*-1*/; n++)
		{
			f.Get(pictorg);

			saveundostep();

			for(int i = 0; i < xres*yres; i++)
				pict[i] = GS(pictorg[i]);

			const int divcol = (numcol > 4) ? (int)(numcol/4) : 1;
			for(int y = 0; y < yres; y++)
				for(int x = 0; x < xres; x++)
				{
					const unsigned int i = x+y*xres;

					if(key == 'i')
					{
#ifdef JITTER_AVI_ORDERED_DITHERING
						const int Diff = ((bayer_dither_pattern[(y^n)&7][(x^n)&7]^(n&63))-31)/divcol;
#else
						const int Diff = (bayer_dither_pattern[y&7][x&7]-31)/divcol;
#endif
						if((int)pict[i]+Diff <= 0)
							pic[current_page][i] = MapColor(0);
						else
							if((int)pict[i]+Diff >= 255)
								pic[current_page][i] = MapColor(255);
							else
								pic[current_page][i] = MapColor(pict[i]+Diff);
					}
					else
					{
						pic[current_page][i] = MapColor(pict[i]);

						if(key == 'o')
						{
#ifdef STUCKI_DITHERING
							const int Err = pict[i]-invcol[pic[current_page][i]]*255/100;
							if(x < xres-1) pict[i+1] += 8*Err/42;
							if(x < xres-2) pict[i+2] += 4*Err/42;
							if(y < yres-1)
							{
								if(x > 1) pict[i-2-xres] += 2*Err/42;
								if(x > 0) pict[i-1-xres] += 4*Err/42;
								pict[i-xres] += 8*Err/42;
								if(x < xres-1) pict[i+1-xres] += 4*Err/42;
								if(x < xres-2) pict[i+2-xres] += 2*Err/42;
								if(y < yres-2)
								{
									if(x > 1) pict[i-2-xres*2] += Err/42;
									if(x > 0) pict[i-1-xres*2] += 2*Err/42;
									pict[i-xres*2] += 4*Err/42;
									if(x < xres-1) pict[i+1-xres*2] += 2*Err/42;
									if(x < xres-2) pict[i+2-xres*2] += Err/42;
								}
							}
#else					
							const int Err = pict[i]-invcol[pic[current_page][i]]*255/100;
							if(x < xres-1) pict[i+1] += 7*Err/16;
							if(y < yres-1)
							{	
								if(x > 0) pict[i-1-xres] += 3*Err/16;
								pict[i-xres] += 5*Err/16;
								if(x < xres-1) pict[i+1-xres] += Err/16;
							}
#endif
						}
					}
				}

			if(n < (int)f.last_frame/*-2*/-1)
				keyboardSpecial(GLUT_KEY_PAGE_UP,0,0); //!! TODO
		}

		free(pict);
		free(pictorg);

		current_page = old_page;
		}
		else if((szFile[slen-1] == 'f') && (szFile[slen-2] == 'd') && (szFile[slen-3] == 'm') && (szFile[slen-4] == 'd'))
		{
		FILE *f = fopen(szFile,"rb");
		if(f == NULL)
			return;

		struct {
			unsigned char name[8];
			unsigned char font_width;
			unsigned char font_width2;
			unsigned char font_height;
			unsigned char font_height2;
			unsigned char end[2];
		} dmdfheader;
		fread(&dmdfheader,sizeof(dmdfheader),1,f);
		if(dmdfheader.name[0] != 'F' || dmdfheader.name[1] != 'P' || dmdfheader.name[2] != 'D' || dmdfheader.name[3] != 'M' || dmdfheader.name[4] != 'D'
			|| dmdfheader.font_width == 0 || dmdfheader.font_height == 0) {
			fclose(f);
			return;
		}

		// code is just copy-pasted again from GIF above!

		if((dmdfheader.font_width != xres) || (dmdfheader.font_height != yres)) {
			fclose(f);
#ifdef WIN32
			MessageBox(NULL,"Resolution does not match the DMD, please change resolution via command line options","LoadPic",MB_OK|MB_ICONSTOP);
#endif
			return;
		}

		unsigned int * const pict = (unsigned int*)malloc(xres*yres*4); 
		
		const unsigned int old_page = current_page;

		for(unsigned int n = 0; n < 95; n++)
		{
		    struct {
			    unsigned char font_width;
			    unsigned char font_width2;
		    } dmdfheader2;
		    fread(&dmdfheader2,sizeof(dmdfheader2),1,f);

			if(dmdfheader2.font_width == xres) {
		        unsigned char *tmp = (unsigned char*)malloc(dmdfheader2.font_width*dmdfheader.font_height);
		        fread(tmp,1,dmdfheader2.font_width*dmdfheader.font_height,f);

				unsigned int o = 0;
			    for(unsigned int y = 0; y < dmdfheader.font_height; ++y)
			        for(unsigned int x = 0; x < dmdfheader2.font_width; ++x,++o)
						pict[o] = (tmp[o] != 7) ? (unsigned int)tmp[o]*255/4 : invcol[stencilcol]*255/100; // 7=transparent

		        free(tmp);
			}

			saveundostep();

			const int divcol = (numcol > 4) ? (int)(numcol/4) : 1;
			for(int y = 0; y < yres; y++)
				for(int x = 0; x < xres; x++)
				{
					const unsigned int i = x+y*xres;

					if(key == 'i')
					{
#ifdef JITTER_AVI_ORDERED_DITHERING
						const int Diff = ((bayer_dither_pattern[(y^n)&7][(x^n)&7]^(n&63))-31)/divcol;
#else
						const int Diff = (bayer_dither_pattern[y&7][x&7]-31)/divcol;
#endif
						if((int)pict[i]+Diff <= 0)
							pic[current_page][i] = MapColor(0);
						else
							if((int)pict[i]+Diff >= 255)
								pic[current_page][i] = MapColor(255);
							else
								pic[current_page][i] = MapColor(pict[i]+Diff);
					}
					else
					{
						pic[current_page][i] = MapColor(pict[i]);

						if(key == 'o')
						{
#ifdef STUCKI_DITHERING
							const int Err = pict[i]-invcol[pic[current_page][i]]*255/100;
							if(x < xres-1) pict[i+1] += 8*Err/42;
							if(x < xres-2) pict[i+2] += 4*Err/42;
							if(y < yres-1)
							{
								if(x > 1) pict[i-2-xres] += 2*Err/42;
								if(x > 0) pict[i-1-xres] += 4*Err/42;
								pict[i-xres] += 8*Err/42;
								if(x < xres-1) pict[i+1-xres] += 4*Err/42;
								if(x < xres-2) pict[i+2-xres] += 2*Err/42;
								if(y < yres-2)
								{
									if(x > 1) pict[i-2-xres*2] += Err/42;
									if(x > 0) pict[i-1-xres*2] += 2*Err/42;
									pict[i-xres*2] += 4*Err/42;
									if(x < xres-1) pict[i+1-xres*2] += 2*Err/42;
									if(x < xres-2) pict[i+2-xres*2] += Err/42;
								}
							}
#else					
							const int Err = pict[i]-invcol[pic[current_page][i]]*255/100;
							if(x < xres-1) pict[i+1] += 7*Err/16;
							if(y < yres-1)
							{	
								if(x > 0) pict[i-1-xres] += 3*Err/16;
								pict[i-xres] += 5*Err/16;
								if(x < xres-1) pict[i+1-xres] += Err/16;
							}
#endif
						}
					}
				}

			if(n < 95/*-2*/-1)
				keyboardSpecial(GLUT_KEY_PAGE_UP,0,0); //!! TODO
		}

		free(pict);

		current_page = old_page;

		fclose(f);
		}
		else
		{
		AVIreader f;
		f.Open(szFile);

		// code is just copy-pasted again from GIF above!

		if((f.xres != xres) || (f.yres != yres)) {
#ifdef WIN32
			MessageBox(NULL,"Resolution does not match the DMD, please change resolution via command line options","LoadPic",MB_OK|MB_ICONSTOP);
#endif
			return;
		}

		unsigned int * const pict = (unsigned int*)malloc(xres*yres*4); 
		
		const unsigned int old_page = current_page;

		for(int n = 0; n < (int)f.last_frame-1; n++)
		{
			f.Get(pict);

			saveundostep();

			for(int i = 0; i < xres*yres; i++)
				pict[i] = GS(pict[i]);

			const int divcol = (numcol > 4) ? (int)(numcol/4) : 1;
			for(int y = 0; y < yres; y++)
				for(int x = 0; x < xres; x++)
				{
					const unsigned int i = x+y*xres;
					const unsigned int i2 = x+(yres-1-y)*xres;

					if(key == 'i')
					{
#ifdef JITTER_AVI_ORDERED_DITHERING
						const int Diff = ((bayer_dither_pattern[(y^n)&7][(x^n)&7]^(n&63))-31)/divcol;
#else
						const int Diff = (bayer_dither_pattern[y&7][x&7]-31)/divcol;
#endif
						if((int)pict[i2]+Diff <= 0)
							pic[current_page][i] = MapColor(0);
						else
							if((int)pict[i2]+Diff >= 255)
								pic[current_page][i] = MapColor(255);
							else
								pic[current_page][i] = MapColor(pict[i2]+Diff);
					}
					else
					{
						pic[current_page][i] = MapColor(pict[i2]);

						if(key == 'o')
						{
#ifdef STUCKI_DITHERING
							const int Err = pict[i2]-invcol[pic[current_page][i]]*255/100;
							if(x < xres-1) pict[i2+1] += 8*Err/42;
							if(x < xres-2) pict[i2+2] += 4*Err/42;
							if(y < yres-1)
							{
								if(x > 1) pict[i2-2-xres] += 2*Err/42;
								if(x > 0) pict[i2-1-xres] += 4*Err/42;
								pict[i2-xres] += 8*Err/42;
								if(x < xres-1) pict[i2+1-xres] += 4*Err/42;
								if(x < xres-2) pict[i2+2-xres] += 2*Err/42;
								if(y < yres-2)
								{
									if(x > 1) pict[i2-2-xres*2] += Err/42;
									if(x > 0) pict[i2-1-xres*2] += 2*Err/42;
									pict[i2-xres*2] += 4*Err/42;
									if(x < xres-1) pict[i2+1-xres*2] += 2*Err/42;
									if(x < xres-2) pict[i2+2-xres*2] += Err/42;
								}
							}
#else					
							const int Err = pict[i2]-invcol[pic[current_page][i]]*255/100;
							if(x < xres-1) pict[i2+1] += 7*Err/16;
							if(y < yres-1)
							{	
								if(x > 0) pict[i2-1-xres] += 3*Err/16;
								pict[i2-xres] += 5*Err/16;
								if(x < xres-1) pict[i2+1-xres] += Err/16;
							}
#endif
						}
					}
				}

			if(n < (int)f.last_frame-2)
				keyboardSpecial(GLUT_KEY_PAGE_UP,0,0); //!! TODO
		}

		free(pict);

		current_page = old_page;
		}
	}
#endif

	// 'f' = flood fill from current cursor position 
	if((key == 'f') && (pic[current_page][cursorx+cursory*xres] != ccol))
	{
		saveundostep();
        recfill(cursorx,cursory);
	}

	// 'p' = copy everything from previous DMD page into the current DMD page
	if((key == 'p') && (current_page > 0))
	{
		saveundostep();
		memcpy(pic[current_page],pic[current_page-1],xres*yres);
	}

	// 't' = delete DMD page
	if((key == 't') && (max_pages > 1))
	{
		for(int i = current_page; i < max_pages-1; i++)
		{
			memcpy(pic[i],pic[i+1],xres*yres);
			memcpy(undoredobuffer[i],undoredobuffer[i+1],xres*yres*redo[i+1]);
			undo[i] = undo[i+1];
			redo[i] = redo[i+1];
		}

		max_pages--;
		if(last_page == max_pages)
			last_page = max_pages-1;
		if(first_page == max_pages)
			first_page = max_pages-1;
		if(current_page == max_pages)
			current_page = max_pages-1;
	}

	// 'g' = insert DMD page
	if(key == 'g')
	{
		if(last_page == max_pages-1)
			last_page++;
		max_pages++;

		pic = (unsigned char**)realloc(pic,sizeof(size_t)*max_pages);
		pic[max_pages-1] = (unsigned char*)malloc(xres*yres);

		undoredobuffer = (unsigned char**)realloc(undoredobuffer,sizeof(size_t)*max_pages);
		undoredobuffer[max_pages-1] = (unsigned char*)malloc(xres*yres*MAX_UNDOREDO);

		undo = (int*)realloc(undo,sizeof(int)*max_pages);
		undo[max_pages-1] = 0;
		redo = (int*)realloc(redo,sizeof(int)*max_pages);
		redo[max_pages-1] = 0;

		current_page++;

		for(int i = max_pages-1; i > current_page; i--)
		{
			memcpy(pic[i],pic[i-1],xres*yres);
			memcpy(undoredobuffer[i],undoredobuffer[i-1],xres*yres*redo[i-1]);
			undo[i] = undo[i-1];
			redo[i] = redo[i-1];
		}

		memset(pic[current_page],0,xres*yres);
	}

	// 'q' = draw pixel with color 0
	if((key == 'q') && (pic[current_page][cursorx+cursory*xres] != 0))
	{
		saveundostep();
		pic[current_page][cursorx+cursory*xres] = 0;
	}
	
	// 'w' = draw pixel with color 1
	if((key == 'w') && (pic[current_page][cursorx+cursory*xres] != 1))
	{
		saveundostep();
		pic[current_page][cursorx+cursory*xres] = 1;
	}

	// 'e' = draw pixel with color 2
	if((key == 'e') && (pic[current_page][cursorx+cursory*xres] != 2))
	{
		saveundostep();
		pic[current_page][cursorx+cursory*xres] = 2;
	}

	// 'r' = draw pixel with color 3
	if((key == 'r') && (pic[current_page][cursorx+cursory*xres] != 3))
	{
		saveundostep();
		pic[current_page][cursorx+cursory*xres] = 3;
	}

	// '1'-'0' = select a color
	if(key == '1')
		ccol = 0;
	if(key == '2')
		ccol = 1;
	if(key == '3')
		ccol = 2;
	if(key == '4')
		ccol = 3;
	if(key == '5')
		ccol = 4;
	if(key == '6')
		ccol = 5;
	if(key == '7')
		ccol = 6;
	if(key == '8')
		ccol = 7;
	if(key == '9')
		ccol = 8;
	if(key == '0')
		ccol = 9;

	// 'SPACE' = lock color to draw with cursor keys
	if(key == ' ')
		lockdraw = !lockdraw;

	// 'z' = undo
	if(key == 'z')
		undostep();

	// 'y' = redo
	if(key == 'y')
		redostep();
}

// Limit Cursorx,y to Window
void checkCursorOutside(const int x, const int y)
{
	cursorx = x*xres/winxres;
	if(cursorx < 0)
		cursorx = 0;
	if(cursorx > xres-1)
		cursorx = xres-1;
	cursory = y*yres/winyres;
	if(cursory < 0)
		cursory = 0;
	if(cursory > yres-1)
		cursory = yres-1;
}

// Move Cursor if no mouse button pressed
void motionPassive(int x, int y)
{
	checkCursorOutside(x,y);
}

// Move Cursor and draw if mouse button pressed
void motion(int x, int y)
{
	checkCursorOutside(x,y);

	if(pic[current_page][cursorx+cursory*xres] != ccol)
	{
		saveundostep();		
		pic[current_page][cursorx+cursory*xres] = ccol;
	}
}

// Move Cursor and draw if mouse button pressed
void mouse(int button, int state, int x, int y)
{
	checkCursorOutside(x,y);

	if((button == GLUT_LEFT_BUTTON) && (state == GLUT_DOWN) && (pic[current_page][cursorx+cursory*xres] != ccol))
	{		
		saveundostep();
		pic[current_page][cursorx+cursory*xres] = ccol;
	}
}

// Idle only does animate the blinking cursor
void idle()
{
	display();
}

void mainMenu(const int value)
{
	switch(value)
	{
	case 1:  keyboard('z',0,0); break;
	case 2:  keyboard('y',0,0); break;
	case 3:  keyboard('+',0,0); break;
	case 4:  keyboard('-',0,0); break;
	case 5:  keyboard('x',0,0); break;
	case 6:  keyboard('v',0,0); break;

	case 7:  keyboard('1',0,0); break;
	case 8:  keyboard('2',0,0); break;
	case 9:  keyboard('3',0,0); break;
	case 10: keyboard('4',0,0); break;
	case 105:keyboard('5',0,0); break;
	case 106:keyboard('6',0,0); break;
	case 107:keyboard('7',0,0); break;
	case 108:keyboard('8',0,0); break;
	case 109:keyboard('9',0,0); break;
	case 110:keyboard('0',0,0); break;
	case 111:keyboardSpecial(GLUT_KEY_F1,0,0); break;
	case 112:keyboardSpecial(GLUT_KEY_F2,0,0); break;
	case 113:keyboardSpecial(GLUT_KEY_F3,0,0); break;
	case 114:keyboardSpecial(GLUT_KEY_F4,0,0); break;
	case 115:keyboardSpecial(GLUT_KEY_F5,0,0); break;
	case 116:keyboardSpecial(GLUT_KEY_F6,0,0); break;

	case 11: keyboard('q',0,0); break;
	case 12: keyboard('w',0,0); break;
	case 13: keyboard('e',0,0); break;
	case 14: keyboard('r',0,0); break;
	case 15: keyboard(' ',0,0); break;
	case 16: keyboard('c',0,0); break;
	case 17: keyboard('f',0,0); break;
	case 18: keyboard('l',0,0); break;
	case 19: keyboard('k',0,0); break;	
	case 20: keyboard('j',0,0); break;
	case 21: keyboard('h',0,0); break;
	case 22: keyboard('o',0,0); break;
	case 23: keyboard('i',0,0); break;
	case 24: keyboard('u',0,0); break;
	case 25: keyboard('s',0,0); break;
	case 26: keyboard('d',0,0); break;
	case 120:keyboard('b',0,0); break;
	case 121:keyboard(',',0,0); break;
	case 27: keyboard('a',0,0); break;
	case 28: keyboardSpecial(GLUT_KEY_PAGE_UP,0,0); break;
	case 29: keyboardSpecial(GLUT_KEY_PAGE_DOWN,0,0); break;
	case 30: keyboard('p',0,0); break;
	case 31: keyboard('t',0,0); break;
	case 32: keyboard('g',0,0); break;
	case 33: keyboard('m',0,0); break;
	case 34: keyboard('n',0,0); break;
	case 41: keyboardSpecial(GLUT_KEY_HOME,0,0); break;
	case 42: keyboardSpecial(GLUT_KEY_END,0,0); break;
	}
}

// Init everything
void initialize()
{
	// init color mapping, color 0 is mapped to DMDmincol, 1..x via linear brightness
	invcol = (unsigned int*)malloc(numcol*4);

	const int mincolindex = (int)((double)(DMDmincol*(numcol-1))/100.0);
	
	for(int i = 0; i < (int)numcol; i++)
	{
		invcol[i] = (unsigned int)((double)(i+mincolindex)/(double)(numcol-1+mincolindex)*100.0);
		if(invcol[i] < DMDmincol)
			invcol[i] = DMDmincol;
	}

	// colored DMD pixel LUTs
	col    = (unsigned int**)malloc(sizeof(size_t)*4);
	col[0] = (unsigned int*) malloc(numcol*4);
	col[1] = (unsigned int*) malloc(numcol*4);
	col[2] = (unsigned int*) malloc(numcol*4);
	col[3] = (unsigned int*) malloc(numcol*4);
	for(int i = 0; i < (int)numcol; i++)
	{
		col[0][i] = ((invcol[i]*255/100)<<8)  | ((invcol[i]/2*255/100)<<16);
		col[1][i] = ((invcol[i]*255/100)<<16) | ((invcol[i]/2*255/100)<<8);
		col[2][i] = ((invcol[i]*255/100)<<16) | ((invcol[i]/5*255/100)<<8) | ((invcol[i]/5*255/100));
		col[3][i] = ((invcol[i]*255/100))     | ((invcol[i]/5*255/100)<<8) | ((invcol[i]/5*255/100)<<16);
	}

	ccol = numcol-1;

	// init pic/undoredo buffers
	pic = (unsigned char**)malloc(sizeof(size_t)*max_pages);
	for(int i = 0; i < max_pages; i++)
	{
		pic[i] = (unsigned char*)malloc(xres*yres);
		memset(pic[i],0,xres*yres);
	}

	undoredobuffer = (unsigned char**)malloc(sizeof(size_t)*max_pages);
	for(int i = 0; i < max_pages; i++)
		undoredobuffer[i] = (unsigned char*)malloc(xres*yres*MAX_UNDOREDO);

	undo = (int*)malloc(sizeof(int)*max_pages);
	memset(undo,0,sizeof(int)*max_pages);
	redo = (int*)malloc(sizeof(int)*max_pages);
	memset(redo,0,sizeof(int)*max_pages);

	// init display buffers
	blitpic = (unsigned int*)malloc(xres*4*yres*4*4);
	memset(blitpic,0,xres*4*yres*4*4);

	TexInit();
}

// Init GLUT and go into mainloop
int main(int argc, char* argv[])
{
	glutInit(&argc, argv);

	if(argc == 6)
	{
		xres      = atoi(argv[1]);
		yres      = atoi(argv[2]);
		numcol    = atoi(argv[3]);
		DMDmincol = atoi(argv[4]);
		stencilcol= atoi(argv[5]);
		printf("using %ux%u with %u colors\ndarkest color maps to %u%% brightness\ncolor %u maps to (optional) stencil",xres,yres,numcol,DMDmincol,stencilcol);
		stencilcol--;
	}
	else
	{
		xres      = 128;
		yres      = 32;
		numcol    = 4;
		DMDmincol = 20;
		stencilcol= 1;
		printf("defaulting to 128x32 with 4 colors\ndarkest color maps to 20%% brightness\ncolor 2 maps to (optional) stencil\n\nuse 'DMDpaint.exe x y c b s' to change defaults");
	}

	if(numcol > 256)
		numcol = 256;
	if(DMDmincol > 100)
		DMDmincol = 100;
	if(stencilcol >= numcol)
		stencilcol = 0;

	winxres = xres*4+1;
	winyres = yres*4+1;

	cursorx = xres/2;
	cursory = yres/2;

	glutInitDisplayMode(GLUT_RGBA | GLUT_DOUBLE);

	glutInitWindowSize(winxres,winyres);
	glutCreateWindow("=DMDpaint=");

	initialize();

	glutDisplayFunc(display);
	glutReshapeFunc(reshape);
	glutKeyboardFunc(keyboard);
	glutSpecialFunc(keyboardSpecial);
	glutMouseFunc(mouse);
	glutMotionFunc(motion);
	glutPassiveMotionFunc(motionPassive);
	glutIdleFunc(idle);

	glutCreateMenu(mainMenu);
	glutAddMenuEntry("Undo 'z'",1);
	glutAddMenuEntry("Redo 'y'",2);
	glutAddMenuEntry("--------",35);
	glutAddMenuEntry("Zoom Window '+'",3);
	glutAddMenuEntry("Zoom Window '-'",4);
	glutAddMenuEntry("--------",36);
	glutAddMenuEntry("Stencil/Sprite Mode 'x'",5);
	glutAddMenuEntry("DMD Drawing Scheme 'v'",6);
	glutAddMenuEntry("DMD Color Scheme 'n'",34);
	glutAddMenuEntry("--------",38);
	glutAddMenuEntry("Clear 'c'",16);
	glutAddMenuEntry("Floodfill 'f'",17);
	glutAddMenuEntry("--------",39);
	glutAddMenuEntry("Load Picture/DMDF (Stucki Dithering) 'l'",18);
	glutAddMenuEntry("Load Picture/DMDF (Ordered Dithering) 'k'",19);
	glutAddMenuEntry("Load Picture/DMDF (No Dithering) 'j'",20);
	glutAddMenuEntry("Load TXT Picture (0..(numcolors-1)) 'h'",21);
	glutAddMenuEntry("Load AVI/GIF/DMDF (Stucki Dithering) 'o'",22);
	glutAddMenuEntry("Load AVI/GIF/DMDF (Ordered Dithering) 'i'",23);
	glutAddMenuEntry("Load AVI/GIF/DMDF (No Dithering) 'u'",24);
	glutAddMenuEntry("--------",44);
	glutAddMenuEntry("Save PGM (As in DMD Preview) 's'",25);
	glutAddMenuEntry("Save PGM (Linearized) 'd'",26);
	glutAddMenuEntry("Save TXT Picture (0..(numcolors-1)) 'a'",27);
	glutAddMenuEntry("Save DMDF from single frame(s) (fontsize=xres/19 x yres/5) 'b'",120);
	glutAddMenuEntry("Save DMDF from all frame(s) (fontsize=xres x yres) ','",121);
	glutAddMenuEntry("--------",40);
	glutAddMenuEntry("Next DMD Page 'PAGE UP'",28);
	glutAddMenuEntry("Previous DMD Page 'PAGE DOWN'",29);
	glutAddMenuEntry("Copy from Previous DMD Page 'p'",30);
	glutAddMenuEntry("--------",45);
	glutAddMenuEntry("Delete DMD Page 't'",31);
	glutAddMenuEntry("Insert DMD Page 'g'",32);
	glutAddMenuEntry("--------",43);
	glutAddMenuEntry("Playback Animation 'm'",33);
	glutAddMenuEntry("Set first DMD Page 'HOME'",41);
	glutAddMenuEntry("Set last DMD Page 'END'",42);
	glutAddMenuEntry("--------",37);
	glutAddMenuEntry("Select Color 01 '1'",7);
	glutAddMenuEntry("Select Color 02 '2'",8);
	glutAddMenuEntry("Select Color 03 '3'",9);
	glutAddMenuEntry("Select Color 04 '4'",10);
	glutAddMenuEntry("Select Color 05 '5'",105);
	glutAddMenuEntry("Select Color 06 '6'",106);
	glutAddMenuEntry("Select Color 07 '7'",107);
	glutAddMenuEntry("Select Color 08 '8'",108);
	glutAddMenuEntry("Select Color 09 '9'",109);
	glutAddMenuEntry("Select Color 10 '0'",110);
	glutAddMenuEntry("Select Color 11 'F1'",111);
	glutAddMenuEntry("Select Color 12 'F2'",112);
	glutAddMenuEntry("Select Color 13 'F3'",113);
	glutAddMenuEntry("Select Color 14 'F4'",114);
	glutAddMenuEntry("Select Color 15 'F5'",115);
	glutAddMenuEntry("Select Color 16 'F6'",116);
	glutAddMenuEntry("Draw Pixel using Color 1 'q'",11);
	glutAddMenuEntry("Draw Pixel using Color 2 'w'",12);
	glutAddMenuEntry("Draw Pixel using Color 3 'e'",13);
	glutAddMenuEntry("Draw Pixel using Color 4 'r'",14);
	glutAddMenuEntry("Draw Lock for Cursor Keys 'SPACE'",15);
	glutAttachMenu(GLUT_RIGHT_BUTTON);
    
	glutMainLoop();

	return 0;
}
